/**
* \file: adit-bdcl.cpp
*
* \version: $Id:$
*
* \release: $Name:$
*
* <brief description>.
* <detailed description>
* \component: Baidu CarLife
*
* \author: P. Govindaraju / RBEB/GM / Pradeepa.Govindaraju@in.bosch.com
*
* \copyright (c) 2016 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
* \see <related items>
*
* \history
*
***********************************************************************/
#include <adit_bdcl.h>

#include <stdio.h>
#include <cmath>
#include <adit_logging.h>
#include "BaiduCoreReceiver.h"
#include "BaiduCoreCallbackDealer.h"

LOG_IMPORT_CONTEXT(bdcl_init)
#define DEFAULT_PROTOCOL_TIMEOUT 8
namespace adit { namespace bdcl {

using namespace std;

bool abdcl::initCmdFlag = false;
bool abdcl::initConditionalFlag = false;
bool abdcl::initAuthenticationFlag = false;
bool abdcl::authenticationResult = false;

std::condition_variable abdcl::cvBdclInit;
std::mutex abdcl::mutBdclInit;
std::condition_variable abdcl::cvInitAuthentication;
std::mutex abdcl::mutInitAuthentication;

IAditBaiduCoreCallbacks* abdcl::mBaiduCoreCallbacks = nullptr;

//GstreamerAdaptation* abdcl::screen = nullptr;

bool abdcl::vrRunning = false;

abdcl::abdcl()
{
    mBaiduCoreCallbacks = nullptr;
    protocolStatusTimeout = DEFAULT_PROTOCOL_TIMEOUT; //value can be overriden by MC using setProtocolStatusTimeOut.
    carlifeStaticsInfo = nullptr; //channel Id has to mention inorder to get uninterrupted baidu session.
    mConfigStructArray = nullptr;
    mCoreReceiver = move(unique_ptr<BaiduCoreReceiver>(new BaiduCoreReceiver()));
    heartBeatThreadId = 0;
    huProtocolStatus = std::cv_status::timeout;
}

 abdcl::~abdcl()
{
    LOGD_DEBUG((bdcl_init,"inside abdcl destructor"));
    CCarLifeLib::getInstance()->setErrorCallback(nullptr);

    if (carlifeStaticsInfo != nullptr)
    {
        delete carlifeStaticsInfo;
        carlifeStaticsInfo = nullptr;
    }
    if (mConfigStructArray != nullptr)
    {
        delete[] mConfigStructArray;
        mConfigStructArray = nullptr;
    }
}

void abdcl::registerAditBaiduCoreCallbacks(IAditBaiduCoreCallbacks* inAditBaiduCoreCallbacks)
{
    mBaiduCoreCallbacks = inAditBaiduCoreCallbacks;
}

void abdcl::setProtocolStatusTimeOut(uint16_t inTimeOut)
{
    this->protocolStatusTimeout = inTimeOut;
}

void abdcl::setCarLifeStaticsInfo(S_STATISTICS_INFO * info)
{
    if (info != nullptr)
    {
        carlifeStaticsInfo = new S_STATISTICS_INFO();
        carlifeStaticsInfo->cuid = info->cuid;
        carlifeStaticsInfo->versionName = info->versionName;
        carlifeStaticsInfo->versionCode = info->versionCode;
        carlifeStaticsInfo->channel = info->channel;
        carlifeStaticsInfo->connectCount = info->connectCount;
        carlifeStaticsInfo->connectSuccessCount = info->connectSuccessCount;
        carlifeStaticsInfo->connectTime = info->connectTime;
        carlifeStaticsInfo->crashLog = info->crashLog;
    }
}

void abdcl::carLifeErrorCallback(void *bdclCtx, carLifeERROR outError)
{
    (void) bdclCtx;
    mBaiduCoreCallbacks->onError(static_cast<bdclErrorCodes>(abs((static_cast<int>(outError))) + 400));
}

bdclInitStatus abdcl::bdclInit(int inAoapAccessoryId, int inAoapDeviceId)
{
    if ((carlifeStaticsInfo != nullptr))
    {

        //initialize CarLife libaray
        CCarLifeLib::getInstance()->carLifeLibInit();

        //support ADB connection
        if (inAoapAccessoryId >= 0 && inAoapDeviceId >= 0)
        {
            if (CONNECTION_PATTERN == ADB_CONNECTION)
            {
                if (0 == CCarLifeLib::getInstance()->connectionSetup(inAoapAccessoryId, inAoapDeviceId, aoap_Device))
                {  //modified by ADIT to pass the accessory and device ID.
                    LOGD_DEBUG((bdcl_init,"command/video/media/tts/vr/control channel connection have been set up"));
                } else
                {
                    LOG_ERROR((bdcl_init,"connection set up failed!"));
                }
            }
        } else
        {
            LOG_ERROR((bdcl_init,"Not valid Accesory and Device Id. inAoapAccessoryId = %d and inAoapDeviceId = %d", inAoapAccessoryId, inAoapDeviceId));
        }

        this->registerBaiduCallBacks();
        //// creating the 5 receiving threads
        if (!mCoreReceiver->createThreads())
        {
        teardown();
        return BDCLInit_NoMemory;
        }
        //passing bdcl Context pointer to CCarLifeLib class
        CCarLifeLib::getInstance()->setBdclContext(this);

        //head unit send protocol version information to mobile device
        initConditionalFlag = false;
        initCmdFlag = false;
        uint16_t protocolCount = 1;
        while (!initCmdFlag)
        {
            int rc;
            LOGD_DEBUG((bdcl_init,"Sending the protocol version trial number = %d", protocolCount));
            rc = CCarLifeLib::getInstance()->cmdHUProtoclVersion(
                    &(this->huProtocolVersion));
            if ((!initConditionalFlag) && (rc == 0))
            {
                std::unique_lock < std::mutex > lckProtocolVersion(mutBdclInit);
                huProtocolStatus = cvBdclInit.wait_for(lckProtocolVersion,
                        std::chrono::seconds(1));
            }

            if (!initCmdFlag)
            {
                if (huProtocolStatus == std::cv_status::no_timeout)
                {
                    LOG_ERROR((bdcl_init,"Protocol Match status Failed due to protocol version mismatch"));
                    teardown();
                    return adit::bdcl::BDCLInit_HUVersionError;
                } else
                {
                    if (protocolCount >= protocolStatusTimeout)
                    {
                        LOG_ERROR((bdcl_init,"Protocol Match status failed after number of trials =  %d", protocolCount));
                        teardown();
                        return adit::bdcl::BDCLInit_TimeoutError;
                    }
                    protocolCount++;
                }
            }
        }

        initAuthenticationFlag = false;
        authenticationResult = false;
        std::cv_status statisticInfoStatus = cv_status::no_timeout;
        CCarLifeLib::getInstance()->cmdStatisticInfo(
                (this->carlifeStaticsInfo));  //0x00018027
        if (!initAuthenticationFlag)
        {
            std::unique_lock < std::mutex > lckStatisticInfo(mutInitAuthentication);
            statisticInfoStatus = cvInitAuthentication.wait_for(lckStatisticInfo,
                    std::chrono::seconds(2));
        }
        if (!authenticationResult)
        {
            LOG_ERROR((bdcl_init,"Wrong Channel Id Statistic Info Failed"));
            if (statisticInfoStatus == std::cv_status::no_timeout)
            {
                teardown();
                return adit::bdcl::BDCLInit_StatisticInfoError;
            } else
            {
                teardown();
                return adit::bdcl::BDCLInit_TimeoutError;
            }
        }
        return BDCLInit_NoError;
    } else
    {
        LOG_ERROR((bdcl_init,"Call setCarLifeStaticsInfo with proper Channel Number Before bdclInit"));
        return BDCLInit_ParamError;

    }
}
bdclInitStatus abdcl:: bdclInit(const char* wFile, const char* rFile)
{
    //open the files here and provide fds to  connectionSetup()
    //todo: close the file appropriately during de-init
    int mWriteFd  = open(wFile, O_NONBLOCK | O_WRONLY);
    int mReadFd   = open(rFile, O_NONBLOCK | O_RDONLY);

    if((mWriteFd < 0) && (mReadFd < 0))
    {
        LOG_ERROR((bdcl_init,"EA Native EndPoint Open failed: %d - %s \n", errno,strerror(errno)));
        return BDCLInit_EAEndpointError;
    }
    if((carlifeStaticsInfo != nullptr))
    {
        {
            std::unique_lock<std::mutex>lckHeartBeat(mutHeartBeatThread);
            heartBeatThreadTerminationFlag = false;
        }
        heartBeatThreadId = 0;


        //initialize CarLife libaray
        CCarLifeLib::getInstance()->carLifeLibInit();  //do we have a separate call for iOS devices

        if ((wFile != nullptr) && (rFile != nullptr))
        {
            if (CONNECTION_PATTERN == ADB_CONNECTION)
            {
                if (0 == CCarLifeLib::getInstance()->connectionSetup(mWriteFd, mReadFd, ios_Device))
                {  //modified by ADIT to pass the accessory and device ID.
                    LOGD_DEBUG((bdcl_init,"command/video/media/tts/vr/control channel connection have been set up"));
                } else
                {
                    LOG_ERROR((bdcl_init,"connection set up failed!"));
                }
            }
        } else
        {
            LOG_ERROR((bdcl_init,"Invalid file path ."));
            return BDCLInit_FileIOError;
        }

        ///// registering the Baidu callbacks for platform items.
        this->registerBaiduCallBacks();
        //// creating the 5 receiving threads
        if (!mCoreReceiver->createThreads())
        {
            LOG_ERROR((bdcl_init," %s teardown BDCLInit_NoMemory", __FUNCTION__));
            teardown();
            return BDCLInit_NoMemory;
        }
        //passing bdcl Context pointer to CCarLifeLib class
        CCarLifeLib::getInstance()->setBdclContext(this);

        //head unit send protocol version information to mobile device
        initConditionalFlag = false;
        initCmdFlag = false;
        uint16_t protocolCount = 1;
        while (!initCmdFlag)
        {
            LOGD_DEBUG((bdcl_init,"Sending the protocol version trial number = %d", protocolCount));
            CCarLifeLib::getInstance()->cmdHUProtoclVersion(&(this->huProtocolVersion));
            if ((!initConditionalFlag)) //check return value of cmdHUProtoclVersion also if ep write is a blocking call
            {
                std::unique_lock < std::mutex > lckProtocolVersion(mutBdclInit);
                huProtocolStatus = cvBdclInit.wait_for(lckProtocolVersion, std::chrono::seconds(2));// SWGIII-27389: workaround decided with Bosch MC to counter the MD's delay in response for retrying protocol version.
            }

            if (!initCmdFlag)
            {
                if (huProtocolStatus == std::cv_status::no_timeout)
                {
                    LOG_ERROR((bdcl_init,"Protocol Match status Failed due to protocol version mismatch"));
                    teardown();
                    return adit::bdcl::BDCLInit_HUVersionError;
                } else
                {
                    if (protocolCount >= protocolStatusTimeout)
                    {
                        LOG_ERROR((bdcl_init,"Protocol Match status failed after number of trials =  %d", protocolCount));
                        teardown();
                        return adit::bdcl::BDCLInit_TimeoutError;
                    }
                    protocolCount++;
                }
            }
        }

        initAuthenticationFlag = false;
        authenticationResult = false;
        std::cv_status statisticInfoStatus = cv_status::no_timeout;
        CCarLifeLib::getInstance()->cmdStatisticInfo(
                (this->carlifeStaticsInfo));  //0x00018027
        if (!initAuthenticationFlag)
        {
            std::unique_lock < std::mutex > lckStatisticInfo(mutInitAuthentication);
            statisticInfoStatus = cvInitAuthentication.wait_for(lckStatisticInfo,
                    std::chrono::seconds(2));
        }
        if (!authenticationResult)
        {
            LOG_ERROR((bdcl_init,"Wrong Channel Id Statistic Info Failed"));
            if (statisticInfoStatus == std::cv_status::no_timeout)
            {
                teardown();
                return adit::bdcl::BDCLInit_StatisticInfoError;
            } else
            {
                teardown();
                return adit::bdcl::BDCLInit_TimeoutError;
            }
        }

        int err;
        err = pthread_create(&heartBeatThreadId, NULL, &abdcl::_VideoHeartBeatThread, this);
        if (err)
        {
            LOG_ERROR((bdcl_init,"error in creating the Video HeartBeat Thread"));
            return BDCLInit_HBThreadFail;
        }

        LOG_INFO((bdcl_init, "init::heartBeatThreadId: %lu", heartBeatThreadId));
        err = pthread_setname_np(heartBeatThreadId, "HeartBeatThread");
        if (err != 0)
            LOG_ERROR((bdcl_init,"error in setting thread Name for Video HeartBeat Thread: %d", err));

        return BDCLInit_NoError;
    }
    else
    {
        LOG_INFO((bdcl_init,"Call setCarLifeStaticsInfo with proper Channel Number Before bdclInit"));
        return BDCLInit_ParamError;

    }
}

void abdcl::registerBaiduCallBacks()
{
    CCarLifeLib::getInstance()->cmdRegisterProtocolVersionMatchStatus(cmdProtocolVersionMatchStatus);
    CCarLifeLib::getInstance()->cmdRegisterMdAuthenResult(cmdRegisterMdAuthenResponse);
    CCarLifeLib::getInstance()->setErrorCallback(carLifeErrorCallback);
}


void abdcl::teardown()
{
    LOGD_DEBUG((bdcl_init, "Entered abdcl::teardown"));
    mCoreReceiver->cleanupThreads();

    {
        std::unique_lock < std::mutex > lckHeartBeat(mutHeartBeatThread);
        heartBeatThreadTerminationFlag = true;
    }
    cvHeartBeatThread.notify_one();
    LOGD_DEBUG(( bdcl_init, "Notify::HeartBeatThreadId: %lu", heartBeatThreadId));
    if (heartBeatThreadId != 0)
    {
        int err = pthread_join(heartBeatThreadId, NULL);
        LOG_INFO(( bdcl_init, "Joining heartBeatThreadId: %d", err));
    }
    LOGD_DEBUG((bdcl_init, "Exit abdcl::teardown"));
}


void abdcl::bdclAbortInit()
{
    LOGD_DEBUG((bdcl_init,"%s() is invoked",__PRETTY_FUNCTION__));
    std::unique_lock<std::mutex> lckProtocolVersion(mutBdclInit);
    initConditionalFlag = true;
    huProtocolStatus = std::cv_status::no_timeout;
    cvBdclInit.notify_one();
    lckProtocolVersion.unlock();
    std::unique_lock < std::mutex > lckStatisticInfo(mutInitAuthentication);
    initAuthenticationFlag = true;
    cvInitAuthentication.notify_one();
}

void abdcl::cmdProtocolVersionMatchStatus(S_PROTOCOL_VERSION_MATCH_SATUS* status)
{
    LOGD_DEBUG((bdcl_init,"cmdProtocolVersionMatchStatus() is invoked"));
    std::unique_lock<std::mutex> lckProtocolVersion(mutBdclInit);
    if (status->matchStatus == 1)
    {
        initCmdFlag = true;
    } else if (status->matchStatus == 2)
    {
        initCmdFlag = false;
    }
    mBaiduCoreCallbacks->onProtocolVersionMatchStatus(initCmdFlag);
    initConditionalFlag = true;
    cvBdclInit.notify_one();
}

void abdcl::cmdRegisterMdAuthenResponse(S_MD_AUTHEN_RESULT* result)
{
    LOGD_DEBUG((bdcl_init,"cmdRegisterMdAuthenResponse() is invoked"));
    mBaiduCoreCallbacks->onAuthenticationResponse(result->authenResult);
    std::unique_lock<std::mutex> lckStatisticInfo(mutInitAuthentication);
    if (result->authenResult)
    {
        authenticationResult = true;
    } else
    {
        authenticationResult = false;
    }
    initAuthenticationFlag = true;
    cvInitAuthentication.notify_one();
}

void abdcl::sendVideoHeartBeat()
{
    std::unique_lock<std::mutex> lckHeartBeat(mutHeartBeatThread);

    while (!heartBeatThreadTerminationFlag)
    {
        if (-1 == CCarLifeLib::getInstance()->sendvideoHeartBeat())
        {
            LOGD_DEBUG(( bdcl_init, "Failed to send heartbeat"));
            heartBeatThreadTerminationFlag = true;
        } else
        {
            cvHeartBeatThread.wait_for(lckHeartBeat, std::chrono::seconds(1));
        }
    }
    pthread_exit(NULL);
}

} } /// namespace close


